<?php
/**
 * Class Payment_Request_Button_States
 *
 * @package WooCommerce\Payments
 */

namespace WCPay\Constants;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * This class provides a list of states for the Payment Request API to match against WooCommerce states.
 * The list is based on libaddressinput: https://github.com/google/libaddressinput,
 * which is used by Chromium based browsers in the native Payment Request address dialog.
 *
 * Known issues/inconsistencies:
 *    1. WC provides a dropdown list of states, but there's no state field in Chrome for the following countries:
 *        AO (Angola), BD (Bangladesh), BG (Bulgaria), BJ (Benin), BO (Bolivia), DO (Dominican Republic),
 *        DZ (Algeria), GH (Ghana), GT (Guatemala), HU (Hungary), KE (Kenya), LA (Laos),
 *        LR (Liberia), MD (Moldova), NA (Namibia), NP (Nepal), PK (Pakistan),
 *        PY (Paraguay), RO (Romania), TZ (Tanzania), UG (Uganda), UM (United States Minor Outlying Islands),
 *        ZA (South Africa), ZM (Zambia).
 *    2. Chrome does not provide a dropdown list of states for 161 countries in total, out of the 249 countries WC supports,
 *        so the countries in which the state field is required by WC, and not provided by the browser are not supported.
 *    3. Chrome does not provide a zip/postal code field for 60 out of the 249 countries WC supports. Only for 5 countries
 *        the zip code field is missing while the state field is present: BS (Bahamas), PA (Panama), SC (Seychelles),
 *        SR (Suriname), TV (Tuvalu). Several other countries provide an optional zip code field.
 *    4. WC expects it, but there's no city field in the Payment Request API for the following:
 *        JP (Japan), MO (Macao), TN (Tunisia), KY (Cayman Islands), GI (Gibraltar).
 *    5. The following countries are not listed in WC:
 *        AC (Ascension Island), IC (Canary Islands), EA (Ceuta & Melilla), CP (Clipperton Island), DG (Diego Garcia),
 *        TA (Tristan da Cunha), XK (Kosovo).
 */
class Payment_Request_Button_States {
	/**
	 * A list of states which is compatible with Chromium based browsers for the Payment Request API.
	 * If the input comes from Chrome, we will always match with `code`, but if the request comes from
	 * Safari or other browsers which don't provide a dropdown list, we can match one of the following:
	 *    - [0] = code (string)
	 *    - [1] = name (string)
	 *    - [2] = localName (string|null)
	 */

	// phpcs:disable
	const STATES = [
		// Afghanistan.
		Country_Code::AFGHANISTAN => [],
		// Angola.
		Country_Code::ANDORRA => [],
		// Argentina.
		Country_Code::ARGENTINA => [
			'C' => [ 'Ciudad Autónoma de Buenos Aires', 'Ciudad Autónoma de Buenos Aires', NULL ],
			'B' => [ 'Buenos Aires', 'Buenos Aires', NULL ],
			'K' => [ 'Catamarca', 'Catamarca', NULL ],
			'H' => [ 'Chaco', 'Chaco', NULL ],
			'U' => [ 'Chubut', 'Chubut', NULL ],
			'X' => [ 'Córdoba', 'Córdoba', NULL ],
			'W' => [ 'Corrientes', 'Corrientes', NULL ],
			'E' => [ 'Entre Ríos', 'Entre Ríos', NULL ],
			'P' => [ 'Formosa', 'Formosa', NULL ],
			'Y' => [ 'Jujuy', 'Jujuy', NULL ],
			'L' => [ 'La Pampa', 'La Pampa', NULL ],
			'F' => [ 'La Rioja', 'La Rioja', NULL ],
			'M' => [ 'Mendoza', 'Mendoza', NULL ],
			'N' => [ 'Misiones', 'Misiones', NULL ],
			'Q' => [ 'Neuquén', 'Neuquén', NULL ],
			'R' => [ 'Río Negro', 'Río Negro', NULL ],
			'A' => [ 'Salta', 'Salta', NULL ],
			'J' => [ 'San Juan', 'San Juan', NULL ],
			'D' => [ 'San Luis', 'San Luis', NULL ],
			'Z' => [ 'Santa Cruz', 'Santa Cruz', NULL ],
			'S' => [ 'Santa Fe', 'Santa Fe', NULL ],
			'G' => [ 'Santiago del Estero', 'Santiago del Estero', NULL ],
			'V' => [ 'Tierra del Fuego', 'Tierra del Fuego', NULL ],
			'T' => [ 'Tucumán', 'Tucumán', NULL ],
		],
		// Austria.
		Country_Code::AUSTRIA => [],
		// Australia.
		Country_Code::AUSTRALIA =>[
			'ACT' => [ 'ACT', 'Australian Capital Territory', NULL ],
			'NSW' => [ 'NSW', 'New South Wales', NULL ],
			'NT'  => [ 'NT', 'Northern Territory', NULL ],
			'QLD' => [ 'QLD', 'Queensland', NULL ],
			'SA'  => [ 'SA', 'South Australia', NULL ],
			'TAS' => [ 'TAS', 'Tasmania', NULL ],
			'VIC' => [ 'VIC', 'Victoria', NULL ],
			'WA'  => [ 'WA', 'Western Australia', NULL ],
			// [ 'JBT', 'Jervis Bay Territory', NULL ],
		],
		// Aland Islands.
		'AX' => [],
		// Bangladesh.
		Country_Code::BANGLADESH => [],
		// Belgium.
		Country_Code::BELGIUM => [],
		// Bulgaria.
		Country_Code::BULGARIA => [],
		// Bahrain.
		Country_Code::BAHRAIN => [],
		// Burundi.
		Country_Code::BURUNDI => [],
		// Benin.
		Country_Code::BENIN => [],
		// Bolivia.
		Country_Code::BOLIVIA => [],
		// Brazil.
		Country_Code::BRAZIL => [
			'AC' => [ 'AC', 'Acre', NULL ],
			'AL' => [ 'AL', 'Alagoas', NULL ],
			'AP' => [ 'AP', 'Amapá', NULL ],
			'AM' => [ 'AM', 'Amazonas', NULL ],
			'BA' => [ 'BA', 'Bahia', NULL ],
			'CE' => [ 'CE', 'Ceará', NULL ],
			'DF' => [ 'DF', 'Distrito Federal', NULL ],
			'ES' => [ 'ES', 'Espírito Santo', NULL ],
			'GO' => [ 'GO', 'Goiás', NULL ],
			'MA' => [ 'MA', 'Maranhão', NULL ],
			'MT' => [ 'MT', 'Mato Grosso', NULL ],
			'MS' => [ 'MS', 'Mato Grosso do Sul', NULL ],
			'MG' => [ 'MG', 'Minas Gerais', NULL ],
			'PA' => [ 'PA', 'Pará', NULL ],
			'PB' => [ 'PB', 'Paraíba', NULL ],
			'PR' => [ 'PR', 'Paraná', NULL ],
			'PE' => [ 'PE', 'Pernambuco', NULL ],
			'PI' => [ 'PI', 'Piauí', NULL ],
			'RJ' => [ 'RJ', 'Rio de Janeiro', NULL ],
			'RN' => [ 'RN', 'Rio Grande do Norte', NULL ],
			'RS' => [ 'RS', 'Rio Grande do Sul', NULL ],
			'RO' => [ 'RO', 'Rondônia', NULL ],
			'RR' => [ 'RR', 'Roraima', NULL ],
			'SC' => [ 'SC', 'Santa Catarina', NULL ],
			'SP' => [ 'SP', 'São Paulo', NULL ],
			'SE' => [ 'SE', 'Sergipe', NULL ],
			'TO' => [ 'TO', 'Tocantins', NULL ],
		],
		// Canada.
		Country_Code::CANADA => [
			'AB' => [ 'AB', 'Alberta', 'Alberta' ],
			'BC' => [ 'BC', 'British Columbia', 'Colombie-Britannique' ],
			'MB' => [ 'MB', 'Manitoba', 'Manitoba' ],
			'NB' => [ 'NB', 'New Brunswick', 'Nouveau-Brunswick' ],
			'NL' => [ 'NL', 'Newfoundland and Labrador', 'Terre-Neuve-et-Labrador' ],
			'NT' => [ 'NT', 'Northwest Territories', 'Territoires du Nord-Ouest' ],
			'NS' => [ 'NS', 'Nova Scotia', 'Nouvelle-Écosse' ],
			'NU' => [ 'NU', 'Nunavut', 'Nunavut' ],
			'ON' => [ 'ON', 'Ontario', 'Ontario' ],
			'PE' => [ 'PE', 'Prince Edward Island', 'Île-du-Prince-Édouard' ],
			'QC' => [ 'QC', 'Quebec', 'Québec' ],
			'SK' => [ 'SK', 'Saskatchewan', 'Saskatchewan' ],
			'YT' => [ 'YT', 'Yukon', 'Yukon' ],
		],
		// Switzerland.
		Country_Code::SWITZERLAND => [],
		// China.
		Country_Code::CHINA => [
			'CN1'  => [ 'Yunnan Sheng', 'Yunnan Sheng', '云南省' ],
			'CN2'  => [ 'Beijing Shi', 'Beijing Shi', '北京市' ],
			'CN3'  => [ 'Tianjin Shi', 'Tianjin Shi', '天津市' ],
			'CN4'  => [ 'Hebei Sheng', 'Hebei Sheng', '河北省' ],
			'CN5'  => [ 'Shanxi Sheng', 'Shanxi Sheng', '山西省' ],
			'CN6'  => [ 'Neimenggu Zizhiqu', 'Neimenggu Zizhiqu', '内蒙古' ],
			'CN7'  => [ 'Liaoning Sheng', 'Liaoning Sheng', '辽宁省' ],
			'CN8'  => [ 'Jilin Sheng', 'Jilin Sheng', '吉林省' ],
			'CN9'  => [ 'Heilongjiang Sheng', 'Heilongjiang Sheng', '黑龙江省' ],
			'CN10' => [ 'Shanghai Shi', 'Shanghai Shi', '上海市' ],
			'CN11' => [ 'Jiangsu Sheng', 'Jiangsu Sheng', '江苏省' ],
			'CN12' => [ 'Zhejiang Sheng', 'Zhejiang Sheng', '浙江省' ],
			'CN13' => [ 'Anhui Sheng', 'Anhui Sheng', '安徽省' ],
			'CN14' => [ 'Fujian Sheng', 'Fujian Sheng', '福建省' ],
			'CN15' => [ 'Jiangxi Sheng', 'Jiangxi Sheng', '江西省' ],
			'CN16' => [ 'Shandong Sheng', 'Shandong Sheng', '山东省' ],
			'CN17' => [ 'Henan Sheng', 'Henan Sheng', '河南省' ],
			'CN18' => [ 'Hubei Sheng', 'Hubei Sheng', '湖北省' ],
			'CN19' => [ 'Hunan Sheng', 'Hunan Sheng', '湖南省' ],
			'CN20' => [ 'Guangdong Sheng', 'Guangdong Sheng', '广东省' ],
			'CN21' => [ 'Guangxi Zhuangzuzizhiqu', 'Guangxi Zhuangzuzizhiqu', '广西' ],
			'CN22' => [ 'Hainan Sheng', 'Hainan Sheng', '海南省' ],
			'CN23' => [ 'Chongqing Shi', 'Chongqing Shi', '重庆市' ],
			'CN24' => [ 'Sichuan Sheng', 'Sichuan Sheng', '四川省' ],
			'CN25' => [ 'Guizhou Sheng', 'Guizhou Sheng', '贵州省' ],
			'CN26' => [ 'Shaanxi Sheng', 'Shaanxi Sheng', '陕西省' ],
			'CN27' => [ 'Gansu Sheng', 'Gansu Sheng', '甘肃省' ],
			'CN28' => [ 'Qinghai Sheng', 'Qinghai Sheng', '青海省' ],
			'CN29' => [ 'Ningxia Huizuzizhiqu', 'Ningxia Huizuzizhiqu', '宁夏' ],
			'CN30' => [ 'Macau', 'Macau', '澳门' ],
			'CN31' => [ 'Xizang Zizhiqu', 'Xizang Zizhiqu', '西藏' ],
			'CN32' => [ 'Xinjiang Weiwuerzizhiqu', 'Xinjiang Weiwuerzizhiqu', '新疆' ],
			// [ 'Taiwan', 'Taiwan', '台湾' ],
			// [ 'Hong Kong', 'Hong Kong', '香港' ],
		],
		// Czech Republic.
		Country_Code::CZECHIA => [],
		// Germany.
		Country_Code::GERMANY => [],
		// Denmark.
		Country_Code::DENMARK => [],
		// Dominican Republic.
		Country_Code::DOMINICAN_REPUBLIC => [],
		// Algeria.
		Country_Code::ALGERIA => [],
		// Estonia.
		Country_Code::ESTONIA => [],
		// Egypt.
		Country_Code::EGYPT => [
			'EGALX' => [ 'Alexandria Governorate', 'Alexandria Governorate', 'الإسكندرية' ],
			'EGASN' => [ 'Aswan Governorate', 'Aswan Governorate', 'أسوان' ],
			'EGAST' => [ 'Asyut Governorate', 'Asyut Governorate', 'أسيوط' ],
			'EGBA'  => [ 'Red Sea Governorate', 'Red Sea Governorate', 'البحر الأحمر' ],
			'EGBH'  => [ 'El Beheira Governorate', 'El Beheira Governorate', 'البحيرة' ],
			'EGBNS' => [ 'Beni Suef Governorate', 'Beni Suef Governorate', 'بني سويف' ],
			'EGC'   => [ 'Cairo Governorate', 'Cairo Governorate', 'القاهرة' ],
			'EGDK'  => [ 'Dakahlia Governorate', 'Dakahlia Governorate', 'الدقهلية' ],
			'EGDT'  => [ 'Damietta Governorate', 'Damietta Governorate', 'دمياط' ],
			'EGFYM' => [ 'Faiyum Governorate', 'Faiyum Governorate', 'الفيوم' ],
			'EGGH'  => [ 'Gharbia Governorate', 'Gharbia Governorate', 'الغربية' ],
			'EGGZ'  => [ 'Giza Governorate', 'Giza Governorate', 'الجيزة' ],
			'EGIS'  => [ 'Ismailia Governorate', 'Ismailia Governorate', 'الإسماعيلية' ],
			'EGJS'  => [ 'South Sinai Governorate', 'South Sinai Governorate', 'جنوب سيناء' ],
			'EGKB'  => [ 'Qalyubia Governorate', 'Qalyubia Governorate', 'القليوبية' ],
			'EGKFS' => [ 'Kafr El Sheikh Governorate', 'Kafr El Sheikh Governorate', 'كفر الشيخ' ],
			'EGKN'  => [ 'Qena Governorate', 'Qena Governorate', 'قنا' ],
			'EGLX'  => [ 'Luxor Governorate', 'Luxor Governorate', 'الأقصر' ],
			'EGMN'  => [ 'Menia Governorate', 'Menia Governorate', 'المنيا' ],
			'EGMNF' => [ 'Menofia Governorate', 'Menofia Governorate', 'المنوفية' ],
			'EGMT'  => [ 'Matrouh Governorate', 'Matrouh Governorate', 'مطروح' ],
			'EGPTS' => [ 'Port Said Governorate', 'Port Said Governorate', 'بورسعيد' ],
			'EGSHG' => [ 'Sohag Governorate', 'Sohag Governorate', 'سوهاج' ],
			'EGSHR' => [ 'Ash Sharqia Governorate', 'Ash Sharqia Governorate', 'الشرقية' ],
			'EGSIN' => [ 'North Sinai Governorate', 'North Sinai Governorate', 'شمال سيناء' ],
			'EGSUZ' => [ 'Suez Governorate', 'Suez Governorate', 'السويس' ],
			'EGWAD' => [ 'New Valley Governorate', 'New Valley Governorate', 'الوادي الجديد' ],
		],
		// Spain.
		Country_Code::SPAIN => [
			'C'  => [ 'A Coruña', 'A Coruña', NULL ],
			'VI' => [ 'Álava', 'Álava', NULL ],
			'AB' => [ 'Albacete', 'Albacete', NULL ],
			'A'  => [ 'Alicante', 'Alicante', NULL ],
			'AL' => [ 'Almería', 'Almería', NULL ],
			'O'  => [ 'Asturias', 'Asturias', NULL ],
			'AV' => [ 'Ávila', 'Ávila', NULL ],
			'BA' => [ 'Badajoz', 'Badajoz', NULL ],
			'PM' => [ 'Balears', 'Balears', NULL ],
			'B'  => [ 'Barcelona', 'Barcelona', NULL ],
			'BU' => [ 'Burgos', 'Burgos', NULL ],
			'CC' => [ 'Cáceres', 'Cáceres', NULL ],
			'CA' => [ 'Cádiz', 'Cádiz', NULL ],
			'S'  => [ 'Cantabria', 'Cantabria', NULL ],
			'CS' => [ 'Castellón', 'Castellón', NULL ],
			'CE' => [ 'Ceuta', 'Ceuta', NULL ],
			'CR' => [ 'Ciudad Real', 'Ciudad Real', NULL ],
			'CO' => [ 'Córdoba', 'Córdoba', NULL ],
			'CU' => [ 'Cuenca', 'Cuenca', NULL ],
			'GI' => [ 'Girona', 'Girona', NULL ],
			'GR' => [ 'Granada', 'Granada', NULL ],
			'GU' => [ 'Guadalajara', 'Guadalajara', NULL ],
			'SS' => [ 'Guipúzcoa', 'Guipúzcoa', NULL ],
			'H'  => [ 'Huelva', 'Huelva', NULL ],
			'HU' => [ 'Huesca', 'Huesca', NULL ],
			'J'  => [ 'Jaén', 'Jaén', NULL ],
			'LO' => [ 'La Rioja', 'La Rioja', NULL ],
			'GC' => [ 'Las Palmas', 'Las Palmas', NULL ],
			'LE' => [ 'León', 'León', NULL ],
			'L'  => [ 'Lleida', 'Lleida', NULL ],
			'LU' => [ 'Lugo', 'Lugo', NULL ],
			'M'  => [ 'Madrid', 'Madrid', NULL ],
			'MA' => [ 'Málaga', 'Málaga', NULL ],
			'ML' => [ 'Melilla', 'Melilla', NULL ],
			'MU' => [ 'Murcia', 'Murcia', NULL ],
			'NA' => [ 'Navarra', 'Navarra', NULL ],
			'OR' => [ 'Ourense', 'Ourense', NULL ],
			'P'  => [ 'Palencia', 'Palencia', NULL ],
			'PO' => [ 'Pontevedra', 'Pontevedra', NULL ],
			'SA' => [ 'Salamanca', 'Salamanca', NULL ],
			'TF' => [ 'Santa Cruz de Tenerife', 'Santa Cruz de Tenerife', NULL ],
			'SG' => [ 'Segovia', 'Segovia', NULL ],
			'SE' => [ 'Sevilla', 'Sevilla', NULL ],
			'SO' => [ 'Soria', 'Soria', NULL ],
			'T'  => [ 'Tarragona', 'Tarragona', NULL ],
			'TE' => [ 'Teruel', 'Teruel', NULL ],
			'TO' => [ 'Toledo', 'Toledo', NULL ],
			'V'  => [ 'Valencia', 'Valencia', NULL ],
			'VA' => [ 'Valladolid', 'Valladolid', NULL ],
			'BI' => [ 'Vizcaya', 'Vizcaya', NULL ],
			'ZA' => [ 'Zamora', 'Zamora', NULL ],
			'Z'  => [ 'Zaragoza', 'Zaragoza', NULL ],
		],
		// Finland.
		Country_Code::FINLAND => [],
		// France.
		Country_Code::FRANCE => [],
		// French Guiana.
		'GF' => [],
		// Ghana.
		Country_Code::GHANA => [],
		// Guadeloupe.
		'GP' => [],
		// Greece.
		Country_Code::GREECE => [],
		// Guatemala.
		Country_Code::GUATEMALA => [],
		// Hong Kong.
		'HK' => [
			'HONG KONG'       => [ 'Hong Kong Island', 'Hong Kong Island', '香港島' ],
			'KOWLOON'         => [ 'Kowloon', 'Kowloon', '九龍' ],
			'NEW TERRITORIES' => [ 'New Territories', 'New Territories', '新界' ],
		],
		// Hungary.
		Country_Code::HUNGARY => [],
		// Indonesia.
		Country_Code::INDONESIA => [
			'AC' => [ 'Aceh', 'Aceh', NULL ],
			'SU' => [ 'Sumatera Utara', 'Sumatera Utara', NULL ],
			'SB' => [ 'Sumatera Barat', 'Sumatera Barat', NULL ],
			'RI' => [ 'Riau', 'Riau', NULL ],
			'KR' => [ 'Kepulauan Riau', 'Kepulauan Riau', NULL ],
			'JA' => [ 'Jambi', 'Jambi', NULL ],
			'SS' => [ 'Sumatera Selatan', 'Sumatera Selatan', NULL ],
			'BB' => [ 'Kepulauan Bangka Belitung', 'Kepulauan Bangka Belitung', NULL ],
			'BE' => [ 'Bengkulu', 'Bengkulu', NULL ],
			'LA' => [ 'Lampung', 'Lampung', NULL ],
			'JK' => [ 'DKI Jakarta', 'DKI Jakarta', NULL ],
			'JB' => [ 'Jawa Barat', 'Jawa Barat', NULL ],
			'BT' => [ 'Banten', 'Banten', NULL ],
			'JT' => [ 'Jawa Tengah', 'Jawa Tengah', NULL ],
			'JI' => [ 'Jawa Timur', 'Jawa Timur', NULL ],
			'YO' => [ 'Daerah Istimewa Yogyakarta', 'Daerah Istimewa Yogyakarta', NULL ],
			'BA' => [ 'Bali', 'Bali', NULL ],
			'NB' => [ 'Nusa Tenggara Barat', 'Nusa Tenggara Barat', NULL ],
			'NT' => [ 'Nusa Tenggara Timur', 'Nusa Tenggara Timur', NULL ],
			'KB' => [ 'Kalimantan Barat', 'Kalimantan Barat', NULL ],
			'KT' => [ 'Kalimantan Tengah', 'Kalimantan Tengah', NULL ],
			'KI' => [ 'Kalimantan Timur', 'Kalimantan Timur', NULL ],
			'KS' => [ 'Kalimantan Selatan', 'Kalimantan Selatan', NULL ],
			'KU' => [ 'Kalimantan Utara', 'Kalimantan Utara', NULL ],
			'SA' => [ 'Sulawesi Utara', 'Sulawesi Utara', NULL ],
			'ST' => [ 'Sulawesi Tengah', 'Sulawesi Tengah', NULL ],
			'SG' => [ 'Sulawesi Tenggara', 'Sulawesi Tenggara', NULL ],
			'SR' => [ 'Sulawesi Barat', 'Sulawesi Barat', NULL ],
			'SN' => [ 'Sulawesi Selatan', 'Sulawesi Selatan', NULL ],
			'GO' => [ 'Gorontalo', 'Gorontalo', NULL ],
			'MA' => [ 'Maluku', 'Maluku', NULL ],
			'MU' => [ 'Maluku Utara', 'Maluku Utara', NULL ],
			'PA' => [ 'Papua', 'Papua', NULL ],
			'PB' => [ 'Papua Barat', 'Papua Barat', NULL ],
			// [ 'Kalimantan Tengah', 'Kalimantan Tengah', NULL ],
			// [ 'Kalimantan Timur', 'Kalimantan Timur', NULL ],
		],
		// Ireland.
		Country_Code::IRELAND => [
			'CW' => [ 'Co. Carlow', 'Co. Carlow', NULL ],
			'CN' => [ 'Co. Cavan', 'Co. Cavan', NULL ],
			'CE' => [ 'Co. Clare', 'Co. Clare', NULL ],
			'CO' => [ 'Co. Cork', 'Co. Cork', NULL ],
			'DL' => [ 'Co. Donegal', 'Co. Donegal', NULL ],
			'D'  => [ 'Co. Dublin', 'Co. Dublin', NULL ],
			'G'  => [ 'Co. Galway', 'Co. Galway', NULL ],
			'KY' => [ 'Co. Kerry', 'Co. Kerry', NULL ],
			'KE' => [ 'Co. Kildare', 'Co. Kildare', NULL ],
			'KK' => [ 'Co. Kilkenny', 'Co. Kilkenny', NULL ],
			'LS' => [ 'Co. Laois', 'Co. Laois', NULL ],
			'LM' => [ 'Co. Leitrim', 'Co. Leitrim', NULL ],
			'LK' => [ 'Co. Limerick', 'Co. Limerick', NULL ],
			'LD' => [ 'Co. Longford', 'Co. Longford', NULL ],
			'LH' => [ 'Co. Louth', 'Co. Louth', NULL ],
			'MO' => [ 'Co. Mayo', 'Co. Mayo', NULL ],
			'MH' => [ 'Co. Meath', 'Co. Meath', NULL ],
			'MN' => [ 'Co. Monaghan', 'Co. Monaghan', NULL ],
			'OY' => [ 'Co. Offaly', 'Co. Offaly', NULL ],
			'RN' => [ 'Co. Roscommon', 'Co. Roscommon', NULL ],
			'SO' => [ 'Co. Sligo', 'Co. Sligo', NULL ],
			'TA' => [ 'Co. Tipperary', 'Co. Tipperary', NULL ],
			'WD' => [ 'Co. Waterford', 'Co. Waterford', NULL ],
			'WH' => [ 'Co. Westmeath', 'Co. Westmeath', NULL ],
			'WX' => [ 'Co. Wexford', 'Co. Wexford', NULL ],
			'WW' => [ 'Co. Wicklow', 'Co. Wicklow', NULL ],
		],
		// Israel.
		Country_Code::ISRAEL => [],
		// Isle of Man.
		'IM' => [],
		// India.
		Country_Code::INDIA => [
			'AP' => [ 'Andhra Pradesh', 'Andhra Pradesh', NULL ],
			'AR' => [ 'Arunachal Pradesh', 'Arunachal Pradesh', NULL ],
			'AS' => [ 'Assam', 'Assam', NULL ],
			'BR' => [ 'Bihar', 'Bihar', NULL ],
			'CT' => [ 'Chhattisgarh', 'Chhattisgarh', NULL ],
			'GA' => [ 'Goa', 'Goa', NULL ],
			'GJ' => [ 'Gujarat', 'Gujarat', NULL ],
			'HR' => [ 'Haryana', 'Haryana', NULL ],
			'HP' => [ 'Himachal Pradesh', 'Himachal Pradesh', NULL ],
			'JK' => [ 'Jammu and Kashmir', 'Jammu & Kashmir', NULL ],
			'JH' => [ 'Jharkhand', 'Jharkhand', NULL ],
			'KA' => [ 'Karnataka', 'Karnataka', NULL ],
			'KL' => [ 'Kerala', 'Kerala', NULL ],
			// 'LA' => __( 'Ladakh', 'woocommerce' ),
			'MP' => [ 'Madhya Pradesh', 'Madhya Pradesh', NULL ],
			'MH' => [ 'Maharashtra', 'Maharashtra', NULL ],
			'MN' => [ 'Manipur', 'Manipur', NULL ],
			'ML' => [ 'Meghalaya', 'Meghalaya', NULL ],
			'MZ' => [ 'Mizoram', 'Mizoram', NULL ],
			'NL' => [ 'Nagaland', 'Nagaland', NULL ],
			'OR' => [ 'Odisha', 'Odisha', NULL ],
			'PB' => [ 'Punjab', 'Punjab', NULL ],
			'RJ' => [ 'Rajasthan', 'Rajasthan', NULL ],
			'SK' => [ 'Sikkim', 'Sikkim', NULL ],
			'TN' => [ 'Tamil Nadu', 'Tamil Nadu', NULL ],
			'TS' => [ 'Telangana', 'Telangana', NULL ],
			'TR' => [ 'Tripura', 'Tripura', NULL ],
			'UK' => [ 'Uttarakhand', 'Uttarakhand', NULL ],
			'UP' => [ 'Uttar Pradesh', 'Uttar Pradesh', NULL ],
			'WB' => [ 'West Bengal', 'West Bengal', NULL ],
			'AN' => [ 'Andaman and Nicobar Islands', 'Andaman & Nicobar', NULL ],
			'CH' => [ 'Chandigarh', 'Chandigarh', NULL ],
			'DN' => [ 'Dadra and Nagar Haveli', 'Dadra & Nagar Haveli', NULL ],
			'DD' => [ 'Daman and Diu', 'Daman & Diu', NULL ],
			'DL' => [ 'Delhi', 'Delhi', NULL ],
			'LD' => [ 'Lakshadweep', 'Lakshadweep', NULL ],
			'PY' => [ 'Puducherry', 'Puducherry', NULL ],
		],
		// Iran.
		Country_Code::IRAN => [
			'KHZ' => [ 'Khuzestan Province', 'Khuzestan Province', 'استان خوزستان' ],
			'THR' => [ 'Tehran Province', 'Tehran Province', 'استان تهران' ],
			'ILM' => [ 'Ilam Province', 'Ilam Province', 'استان ایلام' ],
			'BHR' => [ 'Bushehr Province', 'Bushehr Province', 'استان بوشهر' ],
			'ADL' => [ 'Ardabil Province', 'Ardabil Province', 'استان اردبیل' ],
			'ESF' => [ 'Isfahan Province', 'Isfahan Province', 'استان اصفهان' ],
			'YZD' => [ 'Yazd Province', 'Yazd Province', 'استان یزد' ],
			'KRH' => [ 'Kermanshah Province', 'Kermanshah Province', 'استان کرمانشاه' ],
			'KRN' => [ 'Kerman Province', 'Kerman Province', 'استان کرمان' ],
			'HDN' => [ 'Hamadan Province', 'Hamadan Province', 'استان همدان' ],
			'GZN' => [ 'Qazvin Province', 'Qazvin Province', 'استان قزوین' ],
			'ZJN' => [ 'Zanjan Province', 'Zanjan Province', 'استان زنجان' ],
			'LRS' => [ 'Lorestan Province', 'Lorestan Province', 'استان لرستان' ],
			'ABZ' => [ 'Alborz Province', 'Alborz Province', 'استان البرز' ],
			'EAZ' => [ 'East Azerbaijan Province', 'East Azerbaijan Province', 'استان آذربایجان شرقی' ],
			'WAZ' => [ 'West Azerbaijan Province', 'West Azerbaijan Province', 'استان آذربایجان غربی' ],
			'CHB' => [ 'Chaharmahal and Bakhtiari Province', 'Chaharmahal and Bakhtiari Province', 'استان چهارمحال و بختیاری' ],
			'SKH' => [ 'South Khorasan Province', 'South Khorasan Province', 'استان خراسان جنوبی' ],
			'RKH' => [ 'Razavi Khorasan Province', 'Razavi Khorasan Province', 'استان خراسان رضوی' ],
			'NKH' => [ 'North Khorasan Province', 'North Khorasan Province', 'استان خراسان شمالی' ],
			'SMN' => [ 'Semnan Province', 'Semnan Province', 'استان سمنان' ],
			'FRS' => [ 'Fars Province', 'Fars Province', 'استان فارس' ],
			'QHM' => [ 'Qom Province', 'Qom Province', 'استان قم' ],
			'KRD' => [ 'Kurdistan Province', 'Kurdistan Province', 'استان کردستان' ],
			'KBD' => [ 'Kohgiluyeh and Boyer-Ahmad Province', 'Kohgiluyeh and Boyer-Ahmad Province', 'استان کهگیلویه و بویراحمد' ],
			'GLS' => [ 'Golestan Province', 'Golestan Province', 'استان گلستان' ],
			'GIL' => [ 'Gilan Province', 'Gilan Province', 'استان گیلان' ],
			'MZN' => [ 'Mazandaran Province', 'Mazandaran Province', 'استان مازندران' ],
			'MKZ' => [ 'Markazi Province', 'Markazi Province', 'استان مرکزی' ],
			'HRZ' => [ 'Hormozgan Province', 'Hormozgan Province', 'استان هرمزگان' ],
			'SBN' => [ 'Sistan and Baluchestan Province', 'Sistan and Baluchestan Province', 'استان سیستان و بلوچستان' ],
		],
		// Iceland.
		Country_Code::ICELAND => [],
		// Italy.
		Country_Code::ITALY => [
			'AG' => [ 'AG', 'Agrigento', NULL ],
			'AL' => [ 'AL', 'Alessandria', NULL ],
			'AN' => [ 'AN', 'Ancona', NULL ],
			'AO' => [ 'AO', 'Aosta', NULL ],
			'AR' => [ 'AR', 'Arezzo', NULL ],
			'AP' => [ 'AP', 'Ascoli Piceno', NULL ],
			'AT' => [ 'AT', 'Asti', NULL ],
			'AV' => [ 'AV', 'Avellino', NULL ],
			'BA' => [ 'BA', 'Bari', NULL ],
			'BT' => [ 'BT', 'Barletta-Andria-Trani', NULL ],
			'BL' => [ 'BL', 'Belluno', NULL ],
			'BN' => [ 'BN', 'Benevento', NULL ],
			'BG' => [ 'BG', 'Bergamo', NULL ],
			'BI' => [ 'BI', 'Biella', NULL ],
			'BO' => [ 'BO', 'Bologna', NULL ],
			'BZ' => [ 'BZ', 'Bolzano', NULL ],
			'BS' => [ 'BS', 'Brescia', NULL ],
			'BR' => [ 'BR', 'Brindisi', NULL ],
			'CA' => [ 'CA', 'Cagliari', NULL ],
			'CL' => [ 'CL', 'Caltanissetta', NULL ],
			'CB' => [ 'CB', 'Campobasso', NULL ],
			'CE' => [ 'CE', 'Caserta', NULL ],
			'CT' => [ 'CT', 'Catania', NULL ],
			'CZ' => [ 'CZ', 'Catanzaro', NULL ],
			'CH' => [ 'CH', 'Chieti', NULL ],
			'CO' => [ 'CO', 'Como', NULL ],
			'CS' => [ 'CS', 'Cosenza', NULL ],
			'CR' => [ 'CR', 'Cremona', NULL ],
			'KR' => [ 'KR', 'Crotone', NULL ],
			'CN' => [ 'CN', 'Cuneo', NULL ],
			'EN' => [ 'EN', 'Enna', NULL ],
			'FM' => [ 'FM', 'Fermo', NULL ],
			'FE' => [ 'FE', 'Ferrara', NULL ],
			'FI' => [ 'FI', 'Firenze', NULL ],
			'FG' => [ 'FG', 'Foggia', NULL ],
			'FC' => [ 'FC', 'Forlì-Cesena', NULL ],
			'FR' => [ 'FR', 'Frosinone', NULL ],
			'GE' => [ 'GE', 'Genova', NULL ],
			'GO' => [ 'GO', 'Gorizia', NULL ],
			'GR' => [ 'GR', 'Grosseto', NULL ],
			'IM' => [ 'IM', 'Imperia', NULL ],
			'IS' => [ 'IS', 'Isernia', NULL ],
			'SP' => [ 'SP', 'La Spezia', NULL ],
			'AQ' => [ 'AQ', "L'Aquila", NULL ],
			'LT' => [ 'LT', 'Latina', NULL ],
			'LE' => [ 'LE', 'Lecce', NULL ],
			'LC' => [ 'LC', 'Lecco', NULL ],
			'LI' => [ 'LI', 'Livorno', NULL ],
			'LO' => [ 'LO', 'Lodi', NULL ],
			'LU' => [ 'LU', 'Lucca', NULL ],
			'MC' => [ 'MC', 'Macerata', NULL ],
			'MN' => [ 'MN', 'Mantova', NULL ],
			'MS' => [ 'MS', 'Massa-Carrara', NULL ],
			'MT' => [ 'MT', 'Matera', NULL ],
			'ME' => [ 'ME', 'Messina', NULL ],
			'MI' => [ 'MI', 'Milano', NULL ],
			'MO' => [ 'MO', 'Modena', NULL ],
			'MB' => [ 'MB', 'Monza e Brianza', NULL ],
			'NA' => [ 'NA', 'Napoli', NULL ],
			'NO' => [ 'NO', 'Novara', NULL ],
			'NU' => [ 'NU', 'Nuoro', NULL ],
			'OR' => [ 'OR', 'Oristano', NULL ],
			'PD' => [ 'PD', 'Padova', NULL ],
			'PA' => [ 'PA', 'Palermo', NULL ],
			'PR' => [ 'PR', 'Parma', NULL ],
			'PV' => [ 'PV', 'Pavia', NULL ],
			'PG' => [ 'PG', 'Perugia', NULL ],
			'PU' => [ 'PU', 'Pesaro e Urbino', NULL ],
			'PE' => [ 'PE', 'Pescara', NULL ],
			'PC' => [ 'PC', 'Piacenza', NULL ],
			'PI' => [ 'PI', 'Pisa', NULL ],
			'PT' => [ 'PT', 'Pistoia', NULL ],
			'PN' => [ 'PN', 'Pordenone', NULL ],
			'PZ' => [ 'PZ', 'Potenza', NULL ],
			'PO' => [ 'PO', 'Prato', NULL ],
			'RG' => [ 'RG', 'Ragusa', NULL ],
			'RA' => [ 'RA', 'Ravenna', NULL ],
			'RC' => [ 'RC', 'Reggio Calabria', NULL ],
			'RE' => [ 'RE', 'Reggio Emilia', NULL ],
			'RI' => [ 'RI', 'Rieti', NULL ],
			'RN' => [ 'RN', 'Rimini', NULL ],
			'RM' => [ 'RM', 'Roma', NULL ],
			'RO' => [ 'RO', 'Rovigo', NULL ],
			'SA' => [ 'SA', 'Salerno', NULL ],
			'SS' => [ 'SS', 'Sassari', NULL ],
			'SV' => [ 'SV', 'Savona', NULL ],
			'SI' => [ 'SI', 'Siena', NULL ],
			'SR' => [ 'SR', 'Siracusa', NULL ],
			'SO' => [ 'SO', 'Sondrio', NULL ],
			'SU' => [ 'SU', 'Sud Sardegna', NULL ],
			'TA' => [ 'TA', 'Taranto', NULL ],
			'TE' => [ 'TE', 'Teramo', NULL ],
			'TR' => [ 'TR', 'Terni', NULL ],
			'TO' => [ 'TO', 'Torino', NULL ],
			'TP' => [ 'TP', 'Trapani', NULL ],
			'TN' => [ 'TN', 'Trento', NULL ],
			'TV' => [ 'TV', 'Treviso', NULL ],
			'TS' => [ 'TS', 'Trieste', NULL ],
			'UD' => [ 'UD', 'Udine', NULL ],
			'VA' => [ 'VA', 'Varese', NULL ],
			'VE' => [ 'VE', 'Venezia', NULL ],
			'VB' => [ 'VB', 'Verbano-Cusio-Ossola', NULL ],
			'VC' => [ 'VC', 'Vercelli', NULL ],
			'VR' => [ 'VR', 'Verona', NULL ],
			'VV' => [ 'VV', 'Vibo Valentia', NULL ],
			'VI' => [ 'VI', 'Vicenza', NULL ],
			'VT' => [ 'VT', 'Viterbo', NULL ],
		],
		// Jamaica.
		Country_Code::JAMAICA => [
			'JM-01' => [ 'Kingston', 'Kingston', NULL ],
			'JM-02' => [ 'St. Andrew', 'St. Andrew', NULL ],
			'JM-03' => [ 'St. Thomas', 'St. Thomas', NULL ],
			'JM-04' => [ 'Portland', 'Portland', NULL ],
			'JM-05' => [ 'St. Mary', 'St. Mary', NULL ],
			'JM-06' => [ 'St. Ann', 'St. Ann', NULL ],
			'JM-07' => [ 'Trelawny', 'Trelawny', NULL ],
			'JM-08' => [ 'St. James', 'St. James', NULL ],
			'JM-09' => [ 'Hanover', 'Hanover', NULL ],
			'JM-10' => [ 'Westmoreland', 'Westmoreland', NULL ],
			'JM-11' => [ 'St. Elizabeth', 'St. Elizabeth', NULL ],
			'JM-12' => [ 'Manchester', 'Manchester', NULL ],
			'JM-13' => [ 'Clarendon', 'Clarendon', NULL ],
			'JM-14' => [ 'St. Catherine', 'St. Catherine', NULL ],
		],
		// Japan.
		Country_Code::JAPAN => [
			'JP01' => [ 'Hokkaido', 'Hokkaido', '北海道' ],
			'JP02' => [ 'Aomori', 'Aomori', '青森県' ],
			'JP03' => [ 'Iwate', 'Iwate', '岩手県' ],
			'JP04' => [ 'Miyagi', 'Miyagi', '宮城県' ],
			'JP05' => [ 'Akita', 'Akita', '秋田県' ],
			'JP06' => [ 'Yamagata', 'Yamagata', '山形県' ],
			'JP07' => [ 'Fukushima', 'Fukushima', '福島県' ],
			'JP08' => [ 'Ibaraki', 'Ibaraki', '茨城県' ],
			'JP09' => [ 'Tochigi', 'Tochigi', '栃木県' ],
			'JP10' => [ 'Gunma', 'Gunma', '群馬県' ],
			'JP11' => [ 'Saitama', 'Saitama', '埼玉県' ],
			'JP12' => [ 'Chiba', 'Chiba', '千葉県' ],
			'JP13' => [ 'Tokyo', 'Tokyo', '東京都' ],
			'JP14' => [ 'Kanagawa', 'Kanagawa', '神奈川県' ],
			'JP15' => [ 'Niigata', 'Niigata', '新潟県' ],
			'JP16' => [ 'Toyama', 'Toyama', '富山県' ],
			'JP17' => [ 'Ishikawa', 'Ishikawa', '石川県' ],
			'JP18' => [ 'Fukui', 'Fukui', '福井県' ],
			'JP19' => [ 'Yamanashi', 'Yamanashi', '山梨県' ],
			'JP20' => [ 'Nagano', 'Nagano', '長野県' ],
			'JP21' => [ 'Gifu', 'Gifu', '岐阜県' ],
			'JP22' => [ 'Shizuoka', 'Shizuoka', '静岡県' ],
			'JP23' => [ 'Aichi', 'Aichi', '愛知県' ],
			'JP24' => [ 'Mie', 'Mie', '三重県' ],
			'JP25' => [ 'Shiga', 'Shiga', '滋賀県' ],
			'JP26' => [ 'Kyoto', 'Kyoto', '京都府' ],
			'JP27' => [ 'Osaka', 'Osaka', '大阪府' ],
			'JP28' => [ 'Hyogo', 'Hyogo', '兵庫県' ],
			'JP29' => [ 'Nara', 'Nara', '奈良県' ],
			'JP30' => [ 'Wakayama', 'Wakayama', '和歌山県' ],
			'JP31' => [ 'Tottori', 'Tottori', '鳥取県' ],
			'JP32' => [ 'Shimane', 'Shimane', '島根県' ],
			'JP33' => [ 'Okayama', 'Okayama', '岡山県' ],
			'JP34' => [ 'Hiroshima', 'Hiroshima', '広島県' ],
			'JP35' => [ 'Yamaguchi', 'Yamaguchi', '山口県' ],
			'JP36' => [ 'Tokushima', 'Tokushima', '徳島県' ],
			'JP37' => [ 'Kagawa', 'Kagawa', '香川県' ],
			'JP38' => [ 'Ehime', 'Ehime', '愛媛県' ],
			'JP39' => [ 'Kochi', 'Kochi', '高知県' ],
			'JP40' => [ 'Fukuoka', 'Fukuoka', '福岡県' ],
			'JP41' => [ 'Saga', 'Saga', '佐賀県' ],
			'JP42' => [ 'Nagasaki', 'Nagasaki', '長崎県' ],
			'JP43' => [ 'Kumamoto', 'Kumamoto', '熊本県' ],
			'JP44' => [ 'Oita', 'Oita', '大分県' ],
			'JP45' => [ 'Miyazaki', 'Miyazaki', '宮崎県' ],
			'JP46' => [ 'Kagoshima', 'Kagoshima', '鹿児島県' ],
			'JP47' => [ 'Okinawa', 'Okinawa', '沖縄県' ],
		],
		// Kenya.
		Country_Code::KENYA => [],
		// South Korea.
		Country_Code::SOUTH_KOREA => [],
		// Kuwait.
		Country_Code::KUWAIT => [],
		// Laos.
		Country_Code::LAOS => [],
		// Lebanon.
		Country_Code::LEBANON => [],
		// Sri Lanka.
		'LK' => [],
		// Liberia.
		Country_Code::LIBERIA => [],
		// Luxembourg.
		Country_Code::LUXEMBOURG => [],
		// Moldova.
		Country_Code::MOLDOVA => [],
		// Martinique.
		'MQ' => [],
		// Malta.
		Country_Code::MALTA => [],
		// Mexico.
		Country_Code::MEXICO => [
			'DF' => [ 'CDMX', 'Ciudad de México', NULL ],
			'JA' => [ 'Jal.', 'Jalisco', NULL ],
			'NL' => [ 'N.L.', 'Nuevo León', NULL ],
			'AG' => [ 'Ags.', 'Aguascalientes', NULL ],
			'BC' => [ 'B.C.', 'Baja California', NULL ],
			'BS' => [ 'B.C.S.', 'Baja California Sur', NULL ],
			'CM' => [ 'Camp.', 'Campeche', NULL ],
			'CS' => [ 'Chis.', 'Chiapas', NULL ],
			'CH' => [ 'Chih.', 'Chihuahua', NULL ],
			'CO' => [ 'Coah.', 'Coahuila de Zaragoza', NULL ],
			'CL' => [ 'Col.', 'Colima', NULL ],
			'DG' => [ 'Dgo.', 'Durango', NULL ],
			'GT' => [ 'Gto.', 'Guanajuato', NULL ],
			'GR' => [ 'Gro.', 'Guerrero', NULL ],
			'HG' => [ 'Hgo.', 'Hidalgo', NULL ],
			'MX' => [ 'Méx.', 'Estado de México', NULL ],
			'MI' => [ 'Mich.', 'Michoacán', NULL ],
			'MO' => [ 'Mor.', 'Morelos', NULL ],
			'NA' => [ 'Nay.', 'Nayarit', NULL ],
			'OA' => [ 'Oax.', 'Oaxaca', NULL ],
			'PU' => [ 'Pue.', 'Puebla', NULL ],
			'QT' => [ 'Qro.', 'Querétaro', NULL ],
			'QR' => [ 'Q.R.', 'Quintana Roo', NULL ],
			'SL' => [ 'S.L.P.', 'San Luis Potosí', NULL ],
			'SI' => [ 'Sin.', 'Sinaloa', NULL ],
			'SO' => [ 'Son.', 'Sonora', NULL ],
			'TB' => [ 'Tab.', 'Tabasco', NULL ],
			'TM' => [ 'Tamps.', 'Tamaulipas', NULL ],
			'TL' => [ 'Tlax.', 'Tlaxcala', NULL ],
			'VE' => [ 'Ver.', 'Veracruz', NULL ],
			'YU' => [ 'Yuc.', 'Yucatán', NULL ],
			'ZA' => [ 'Zac.', 'Zacatecas', NULL ],
		],
		// Malaysia.
		Country_Code::MALAYSIA => [
			'JHR' => [ 'Johor', 'Johor', NULL ],
			'KDH' => [ 'Kedah', 'Kedah', NULL ],
			'KTN' => [ 'Kelantan', 'Kelantan', NULL ],
			'LBN' => [ 'Labuan', 'Labuan', NULL ],
			'MLK' => [ 'Melaka', 'Melaka', NULL ],
			'NSN' => [ 'Negeri Sembilan', 'Negeri Sembilan', NULL ],
			'PHG' => [ 'Pahang', 'Pahang', NULL ],
			'PNG' => [ 'Pulau Pinang', 'Pulau Pinang', NULL ],
			'PRK' => [ 'Perak', 'Perak', NULL ],
			'PLS' => [ 'Perlis', 'Perlis', NULL ],
			'SBH' => [ 'Sabah', 'Sabah', NULL ],
			'SWK' => [ 'Sarawak', 'Sarawak', NULL ],
			'SGR' => [ 'Selangor', 'Selangor', NULL ],
			'TRG' => [ 'Terengganu', 'Terengganu', NULL ],
			'PJY' => [ 'Putrajaya', 'Putrajaya', NULL ],
			'KUL' => [ 'Kuala Lumpur', 'Kuala Lumpur', NULL ],
		],
		// Mozambique.
		Country_Code::MOZAMBIQUE => [
			'MZP'   => [ 'Cabo Delgado', 'Cabo Delgado', NULL ],
			'MZG'   => [ 'Gaza', 'Gaza', NULL ],
			'MZI'   => [ 'Inhambane', 'Inhambane', NULL ],
			'MZB'   => [ 'Manica', 'Manica', NULL ],
			'MZL'   => [ 'Maputo', 'Maputo', NULL ],
			'MZMPM' => [ 'Cidade de Maputo', 'Cidade de Maputo', NULL ],
			'MZN'   => [ 'Nampula', 'Nampula', NULL ],
			'MZA'   => [ 'Niassa', 'Niassa', NULL ],
			'MZS'   => [ 'Sofala', 'Sofala', NULL ],
			'MZT'   => [ 'Tete', 'Tete', NULL ],
			'MZQ'   => [ 'Zambezia', 'Zambezia', NULL ],
		],
		// Namibia.
		Country_Code::NAMIBIA => [],
		// Nigeria.
		Country_Code::NIGERIA => [
			'AB' => [ 'Abia', 'Abia', NULL ],
			'FC' => [ 'Federal Capital Territory', 'Federal Capital Territory', NULL ],
			'AD' => [ 'Adamawa', 'Adamawa', NULL ],
			'AK' => [ 'Akwa Ibom', 'Akwa Ibom', NULL ],
			'AN' => [ 'Anambra', 'Anambra', NULL ],
			'BA' => [ 'Bauchi', 'Bauchi', NULL ],
			'BY' => [ 'Bayelsa', 'Bayelsa', NULL ],
			'BE' => [ 'Benue', 'Benue', NULL ],
			'BO' => [ 'Borno', 'Borno', NULL ],
			'CR' => [ 'Cross River', 'Cross River', NULL ],
			'DE' => [ 'Delta', 'Delta', NULL ],
			'EB' => [ 'Ebonyi', 'Ebonyi', NULL ],
			'ED' => [ 'Edo', 'Edo', NULL ],
			'EK' => [ 'Ekiti', 'Ekiti', NULL ],
			'EN' => [ 'Enugu', 'Enugu', NULL ],
			'GO' => [ 'Gombe', 'Gombe', NULL ],
			'IM' => [ 'Imo', 'Imo', NULL ],
			'JI' => [ 'Jigawa', 'Jigawa', NULL ],
			'KD' => [ 'Kaduna', 'Kaduna', NULL ],
			'KN' => [ 'Kano', 'Kano', NULL ],
			'KT' => [ 'Katsina', 'Katsina', NULL ],
			'KE' => [ 'Kebbi', 'Kebbi', NULL ],
			'KO' => [ 'Kogi', 'Kogi', NULL ],
			'KW' => [ 'Kwara', 'Kwara', NULL ],
			'LA' => [ 'Lagos', 'Lagos', NULL ],
			'NA' => [ 'Nasarawa', 'Nasarawa', NULL ],
			'NI' => [ 'Niger', 'Niger', NULL ],
			'OG' => [ 'Ogun State', 'Ogun State', NULL ],
			'ON' => [ 'Ondo', 'Ondo', NULL ],
			'OS' => [ 'Osun', 'Osun', NULL ],
			'OY' => [ 'Oyo', 'Oyo', NULL ],
			'PL' => [ 'Plateau', 'Plateau', NULL ],
			'RI' => [ 'Rivers', 'Rivers', NULL ],
			'SO' => [ 'Sokoto', 'Sokoto', NULL ],
			'TA' => [ 'Taraba', 'Taraba', NULL ],
			'YO' => [ 'Yobe', 'Yobe', NULL ],
			'ZA' => [ 'Zamfara', 'Zamfara', NULL ],
		],
		// Netherlands.
		Country_Code::NETHERLANDS => [],
		// Norway.
		Country_Code::NORWAY => [],
		// Nepal.
		Country_Code::NEPAL => [],
		// New Zealand.
		Country_Code::NEW_ZEALAND => [],
		// Peru.
		Country_Code::PERU => [
			'CAL' => [ 'Callao', 'Callao', NULL ],
			'LMA' => [ 'Municipalidad Metropolitana de Lima', 'Municipalidad Metropolitana de Lima', NULL ],
			'AMA' => [ 'Amazonas', 'Amazonas', NULL ],
			'ANC' => [ 'Áncash', 'Áncash', NULL ],
			'APU' => [ 'Apurímac', 'Apurímac', NULL ],
			'ARE' => [ 'Arequipa', 'Arequipa', NULL ],
			'AYA' => [ 'Ayacucho', 'Ayacucho', NULL ],
			'CAJ' => [ 'Cajamarca', 'Cajamarca', NULL ],
			'CUS' => [ 'Cuzco', 'Cuzco', NULL ],
			'HUV' => [ 'Huancavelica', 'Huancavelica', NULL ],
			'HUC' => [ 'Huánuco', 'Huánuco', NULL ],
			'ICA' => [ 'Ica', 'Ica', NULL ],
			'JUN' => [ 'Junín', 'Junín', NULL ],
			'LAL' => [ 'La Libertad', 'La Libertad', NULL ],
			'LAM' => [ 'Lambayeque', 'Lambayeque', NULL ],
			'LIM' => [ 'Gobierno Regional de Lima', 'Gobierno Regional de Lima', NULL ],
			'LOR' => [ 'Loreto', 'Loreto', NULL ],
			'MDD' => [ 'Madre de Dios', 'Madre de Dios', NULL ],
			'MOQ' => [ 'Moquegua', 'Moquegua', NULL ],
			'PAS' => [ 'Pasco', 'Pasco', NULL ],
			'PIU' => [ 'Piura', 'Piura', NULL ],
			'PUN' => [ 'Puno', 'Puno', NULL ],
			'SAM' => [ 'San Martín', 'San Martín', NULL ],
			'TAC' => [ 'Tacna', 'Tacna', NULL ],
			'TUM' => [ 'Tumbes', 'Tumbes', NULL ],
			'UCA' => [ 'Ucayali', 'Ucayali', NULL ],
		],
		// Philippines.
		Country_Code::PHILIPPINES => [
			'ABR' => [ 'Abra', 'Abra', NULL ],
			'AGN' => [ 'Agusan del Norte', 'Agusan del Norte', NULL ],
			'AGS' => [ 'Agusan del Sur', 'Agusan del Sur', NULL ],
			'AKL' => [ 'Aklan', 'Aklan', NULL ],
			'ALB' => [ 'Albay', 'Albay', NULL ],
			'ANT' => [ 'Antique', 'Antique', NULL ],
			'APA' => [ 'Apayao', 'Apayao', NULL ],
			'AUR' => [ 'Aurora', 'Aurora', NULL ],
			'BAS' => [ 'Basilan', 'Basilan', NULL ],
			'BAN' => [ 'Bataan', 'Bataan', NULL ],
			'BTN' => [ 'Batanes', 'Batanes', NULL ],
			'BTG' => [ 'Batangas', 'Batangas', NULL ],
			'BEN' => [ 'Benguet', 'Benguet', NULL ],
			'BIL' => [ 'Biliran', 'Biliran', NULL ],
			'BOH' => [ 'Bohol', 'Bohol', NULL ],
			'BUK' => [ 'Bukidnon', 'Bukidnon', NULL ],
			'BUL' => [ 'Bulacan', 'Bulacan', NULL ],
			'CAG' => [ 'Cagayan', 'Cagayan', NULL ],
			'CAN' => [ 'Camarines Norte', 'Camarines Norte', NULL ],
			'CAS' => [ 'Camarines Sur', 'Camarines Sur', NULL ],
			'CAM' => [ 'Camiguin', 'Camiguin', NULL ],
			'CAP' => [ 'Capiz', 'Capiz', NULL ],
			'CAT' => [ 'Catanduanes', 'Catanduanes', NULL ],
			'CAV' => [ 'Cavite', 'Cavite', NULL ],
			'CEB' => [ 'Cebu', 'Cebu', NULL ],
			'COM' => [ 'Compostela Valley', 'Compostela Valley', NULL ],
			'NCO' => [ 'Cotabato', 'Cotabato', NULL ],
			'DAV' => [ 'Davao del Norte', 'Davao del Norte', NULL ],
			'DAS' => [ 'Davao del Sur', 'Davao del Sur', NULL ],
			'DAC' => [ 'Davao Occidental', 'Davao Occidental', NULL ],
			'DAO' => [ 'Davao Oriental', 'Davao Oriental', NULL ],
			'DIN' => [ 'Dinagat Islands', 'Dinagat Islands', NULL ],
			'EAS' => [ 'Eastern Samar', 'Eastern Samar', NULL ],
			'GUI' => [ 'Guimaras', 'Guimaras', NULL ],
			'IFU' => [ 'Ifugao', 'Ifugao', NULL ],
			'ILN' => [ 'Ilocos Norte', 'Ilocos Norte', NULL ],
			'ILS' => [ 'Ilocos Sur', 'Ilocos Sur', NULL ],
			'ILI' => [ 'Iloilo', 'Iloilo', NULL ],
			'ISA' => [ 'Isabela', 'Isabela', NULL ],
			'KAL' => [ 'Kalinga', 'Kalinga', NULL ],
			'LUN' => [ 'La Union', 'La Union', NULL ],
			'LAG' => [ 'Laguna', 'Laguna', NULL ],
			'LAN' => [ 'Lanao del Norte', 'Lanao del Norte', NULL ],
			'LAS' => [ 'Lanao del Sur', 'Lanao del Sur', NULL ],
			'LEY' => [ 'Leyte', 'Leyte', NULL ],
			'MAG' => [ 'Maguindanao', 'Maguindanao', NULL ],
			'MAD' => [ 'Marinduque', 'Marinduque', NULL ],
			'MAS' => [ 'Masbate', 'Masbate', NULL ],
			'MSC' => [ 'Misamis Occidental', 'Misamis Occidental', NULL ],
			'MSR' => [ 'Misamis Oriental', 'Misamis Oriental', NULL ],
			'MOU' => [ 'Mountain Province', 'Mountain Province', NULL ],
			'NEC' => [ 'Negros Occidental', 'Negros Occidental', NULL ],
			'NER' => [ 'Negros Oriental', 'Negros Oriental', NULL ],
			'NSA' => [ 'Northern Samar', 'Northern Samar', NULL ],
			'NUE' => [ 'Nueva Ecija', 'Nueva Ecija', NULL ],
			'NUV' => [ 'Nueva Vizcaya', 'Nueva Vizcaya', NULL ],
			'MDC' => [ 'Mindoro Occidental', 'Mindoro Occidental', NULL ],
			'MDR' => [ 'Mindoro Oriental', 'Mindoro Oriental', NULL ],
			'PLW' => [ 'Palawan', 'Palawan', NULL ],
			'PAM' => [ 'Pampanga', 'Pampanga', NULL ],
			'PAN' => [ 'Pangasinan', 'Pangasinan', NULL ],
			'QUE' => [ 'Quezon Province', 'Quezon Province', NULL ],
			'QUI' => [ 'Quirino', 'Quirino', NULL ],
			'RIZ' => [ 'Rizal', 'Rizal', NULL ],
			'ROM' => [ 'Romblon', 'Romblon', NULL ],
			'WSA' => [ 'Samar', 'Samar', NULL ],
			'SAR' => [ 'Sarangani', 'Sarangani', NULL ],
			'SIQ' => [ 'Siquijor', 'Siquijor', NULL ],
			'SOR' => [ 'Sorsogon', 'Sorsogon', NULL ],
			'SCO' => [ 'South Cotabato', 'South Cotabato', NULL ],
			'SLE' => [ 'Southern Leyte', 'Southern Leyte', NULL ],
			'SUK' => [ 'Sultan Kudarat', 'Sultan Kudarat', NULL ],
			'SLU' => [ 'Sulu', 'Sulu', NULL ],
			'SUN' => [ 'Surigao del Norte', 'Surigao del Norte', NULL ],
			'SUR' => [ 'Surigao del Sur', 'Surigao del Sur', NULL ],
			'TAR' => [ 'Tarlac', 'Tarlac', NULL ],
			'TAW' => [ 'Tawi-Tawi', 'Tawi-Tawi', NULL ],
			'ZMB' => [ 'Zambales', 'Zambales', NULL ],
			'ZAN' => [ 'Zamboanga del Norte', 'Zamboanga del Norte', NULL ],
			'ZAS' => [ 'Zamboanga del Sur', 'Zamboanga del Sur', NULL ],
			'ZSI' => [ 'Zamboanga Sibuguey', 'Zamboanga Sibuguey', NULL ],
			'00'  => [ 'Metro Manila', 'Metro Manila', NULL ],
		],
		// Pakistan.
		Country_Code::PAKISTAN => [],
		// Poland.
		Country_Code::POLAND => [],
		// Puerto Rico.
		'PR' => [],
		// Portugal.
		Country_Code::PORTUGAL => [],
		// Paraguay.
		Country_Code::PARAGUAY => [],
		// Reunion.
		'RE' => [],
		// Romania.
		Country_Code::ROMANIA => [],
		// Serbia.
		Country_Code::SERBIA => [],
		// Sweden.
		Country_Code::SWEDEN => [],
		// Singapore.
		Country_Code::SINGAPORE => [],
		// Slovenia.
		Country_Code::SLOVENIA => [],
		// Slovakia.
		Country_Code::SLOVAKIA => [],
		// Thailand.
		Country_Code::THAILAND => [
			'TH-37' => [ 'Amnat Charoen', 'Amnat Charoen', 'อำนาจเจริญ' ],
			'TH-15' => [ 'Ang Thong', 'Ang Thong', 'อ่างทอง' ],
			'TH-14' => [ 'Phra Nakhon Si Ayutthaya', 'Phra Nakhon Si Ayutthaya', 'พระนครศรีอยุธยา' ],
			'TH-10' => [ 'Bangkok', 'Bangkok', 'กรุงเทพมหานคร' ],
			'TH-38' => [ 'Bueng Kan', 'Bueng Kan', 'จังหวัด บึงกาฬ' ],
			'TH-31' => [ 'Buri Ram', 'Buri Ram', 'บุรีรัมย์' ],
			'TH-24' => [ 'Chachoengsao', 'Chachoengsao', 'ฉะเชิงเทรา' ],
			'TH-18' => [ 'Chai Nat', 'Chai Nat', 'ชัยนาท' ],
			'TH-36' => [ 'Chaiyaphum', 'Chaiyaphum', 'ชัยภูมิ' ],
			'TH-22' => [ 'Chanthaburi', 'Chanthaburi', 'จันทบุรี' ],
			'TH-50' => [ 'Chiang Rai', 'Chiang Rai', 'เชียงราย' ],
			'TH-57' => [ 'Chiang Mai', 'Chiang Mai', 'เชียงใหม่' ],
			'TH-20' => [ 'Chon Buri', 'Chon Buri', 'ชลบุรี' ],
			'TH-86' => [ 'Chumpon', 'Chumpon', 'ชุมพร' ],
			'TH-46' => [ 'Kalasin', 'Kalasin', 'กาฬสินธุ์' ],
			'TH-62' => [ 'Kamphaeng Phet', 'Kamphaeng Phet', 'กำแพงเพชร' ],
			'TH-71' => [ 'Kanchanaburi', 'Kanchanaburi', 'กาญจนบุรี' ],
			'TH-40' => [ 'Khon Kaen', 'Khon Kaen', 'ขอนแก่น' ],
			'TH-81' => [ 'Krabi', 'Krabi', 'กระบี่' ],
			'TH-52' => [ 'Lampang', 'Lampang', 'ลำปาง' ],
			'TH-51' => [ 'Lamphun', 'Lamphun', 'ลำพูน' ],
			'TH-42' => [ 'Loei', 'Loei', 'เลย' ],
			'TH-16' => [ 'Lop Buri', 'Lop Buri', 'ลพบุรี' ],
			'TH-58' => [ 'Mae Hong Son', 'Mae Hong Son', 'แม่ฮ่องสอน' ],
			'TH-44' => [ 'Maha Sarakham', 'Maha Sarakham', 'มหาสารคาม' ],
			'TH-49' => [ 'Mukdahan', 'Mukdahan', 'มุกดาหาร' ],
			'TH-26' => [ 'Nakhon Nayok', 'Nakhon Nayok', 'นครนายก' ],
			'TH-73' => [ 'Nakhon Pathom', 'Nakhon Pathom', 'นครปฐม' ],
			'TH-48' => [ 'Nakhon Phanom', 'Nakhon Phanom', 'นครพนม' ],
			'TH-30' => [ 'Nakhon Ratchasima', 'Nakhon Ratchasima', 'นครราชสีมา' ],
			'TH-60' => [ 'Nakhon Sawan', 'Nakhon Sawan', 'นครสวรรค์' ],
			'TH-80' => [ 'Nakhon Si Thammarat', 'Nakhon Si Thammarat', 'นครศรีธรรมราช' ],
			'TH-55' => [ 'Nan', 'Nan', 'น่าน' ],
			'TH-96' => [ 'Narathiwat', 'Narathiwat', 'นราธิวาส' ],
			'TH-39' => [ 'Nong Bua Lam Phu', 'Nong Bua Lam Phu', 'หนองบัวลำภู' ],
			'TH-43' => [ 'Nong Khai', 'Nong Khai', 'หนองคาย' ],
			'TH-12' => [ 'Nonthaburi', 'Nonthaburi', 'นนทบุรี' ],
			'TH-13' => [ 'Pathum Thani', 'Pathum Thani', 'ปทุมธานี' ],
			'TH-94' => [ 'Pattani', 'Pattani', 'ปัตตานี' ],
			'TH-82' => [ 'Phang Nga', 'Phang Nga', 'พังงา' ],
			'TH-93' => [ 'Phattalung', 'Phattalung', 'พัทลุง' ],
			'TH-56' => [ 'Phayao', 'Phayao', 'พะเยา' ],
			'TH-67' => [ 'Phetchabun', 'Phetchabun', 'เพชรบูรณ์' ],
			'TH-76' => [ 'Phetchaburi', 'Phetchaburi', 'เพชรบุรี' ],
			'TH-66' => [ 'Phichit', 'Phichit', 'พิจิตร' ],
			'TH-65' => [ 'Phitsanulok', 'Phitsanulok', 'พิษณุโลก' ],
			'TH-54' => [ 'Phrae', 'Phrae', 'แพร่' ],
			'TH-83' => [ 'Phuket', 'Phuket', 'ภูเก็ต' ],
			'TH-25' => [ 'Prachin Buri', 'Prachin Buri', 'ปราจีนบุรี' ],
			'TH-77' => [ 'Prachuap Khiri Khan', 'Prachuap Khiri Khan', 'ประจวบคีรีขันธ์' ],
			'TH-85' => [ 'Ranong', 'Ranong', 'ระนอง' ],
			'TH-70' => [ 'Ratchaburi', 'Ratchaburi', 'ราชบุรี' ],
			'TH-21' => [ 'Rayong', 'Rayong', 'ระยอง' ],
			'TH-45' => [ 'Roi Et', 'Roi Et', 'ร้อยเอ็ด' ],
			'TH-27' => [ 'Sa Kaeo', 'Sa Kaeo', 'สระแก้ว' ],
			'TH-47' => [ 'Sakon Nakhon', 'Sakon Nakhon', 'สกลนคร' ],
			'TH-11' => [ 'Samut Prakan', 'Samut Prakan', 'สมุทรปราการ' ],
			'TH-74' => [ 'Samut Sakhon', 'Samut Sakhon', 'สมุทรสาคร' ],
			'TH-75' => [ 'Samut Songkhram', 'Samut Songkhram', 'สมุทรสงคราม' ],
			'TH-19' => [ 'Saraburi', 'Saraburi', 'สระบุรี' ],
			'TH-91' => [ 'Satun', 'Satun', 'สตูล' ],
			'TH-17' => [ 'Sing Buri', 'Sing Buri', 'สิงห์บุรี' ],
			'TH-33' => [ 'Si Sa Ket', 'Si Sa Ket', 'ศรีสะเกษ' ],
			'TH-90' => [ 'Songkhla', 'Songkhla', 'สงขลา' ],
			'TH-64' => [ 'Sukhothai', 'Sukhothai', 'สุโขทัย' ],
			'TH-72' => [ 'Suphanburi', 'Suphanburi', 'สุพรรณบุรี' ],
			'TH-84' => [ 'Surat Thani', 'Surat Thani', 'สุราษฎร์ธานี' ],
			'TH-32' => [ 'Surin', 'Surin', 'สุรินทร์' ],
			'TH-63' => [ 'Tak', 'Tak', 'ตาก' ],
			'TH-92' => [ 'Trang', 'Trang', 'ตรัง' ],
			'TH-23' => [ 'Trat', 'Trat', 'ตราด' ],
			'TH-34' => [ 'Ubon Ratchathani', 'Ubon Ratchathani', 'อุบลราชธานี' ],
			'TH-41' => [ 'Udon Thani', 'Udon Thani', 'อุดรธานี' ],
			'TH-61' => [ 'Uthai Thani', 'Uthai Thani', 'อุทัยธานี' ],
			'TH-53' => [ 'Uttaradit', 'Uttaradit', 'อุตรดิตถ์' ],
			'TH-95' => [ 'Yala', 'Yala', 'ยะลา' ],
			'TH-35' => [ 'Yasothon', 'Yasothon', 'ยโสธร' ],
		],
		// Turkey.
		Country_Code::TURKEY => [
			'TR01' => [ 'Adana', 'Adana', NULL ],
			'TR02' => [ 'Adıyaman', 'Adıyaman', NULL ],
			'TR03' => [ 'Afyon', 'Afyon', NULL ],
			'TR04' => [ 'Ağrı', 'Ağrı', NULL ],
			'TR05' => [ 'Amasya', 'Amasya', NULL ],
			'TR06' => [ 'Ankara', 'Ankara', NULL ],
			'TR07' => [ 'Antalya', 'Antalya', NULL ],
			'TR08' => [ 'Artvin', 'Artvin', NULL ],
			'TR09' => [ 'Aydın', 'Aydın', NULL ],
			'TR10' => [ 'Balıkesir', 'Balıkesir', NULL ],
			'TR11' => [ 'Bilecik', 'Bilecik', NULL ],
			'TR12' => [ 'Bingöl', 'Bingöl', NULL ],
			'TR13' => [ 'Bitlis', 'Bitlis', NULL ],
			'TR14' => [ 'Bolu', 'Bolu', NULL ],
			'TR15' => [ 'Burdur', 'Burdur', NULL ],
			'TR16' => [ 'Bursa', 'Bursa', NULL ],
			'TR17' => [ 'Çanakkale', 'Çanakkale', NULL ],
			'TR18' => [ 'Çankırı', 'Çankırı', NULL ],
			'TR19' => [ 'Çorum', 'Çorum', NULL ],
			'TR20' => [ 'Denizli', 'Denizli', NULL ],
			'TR21' => [ 'Diyarbakır', 'Diyarbakır', NULL ],
			'TR22' => [ 'Edirne', 'Edirne', NULL ],
			'TR23' => [ 'Elazığ', 'Elazığ', NULL ],
			'TR24' => [ 'Erzincan', 'Erzincan', NULL ],
			'TR25' => [ 'Erzurum', 'Erzurum', NULL ],
			'TR26' => [ 'Eskişehir', 'Eskişehir', NULL ],
			'TR27' => [ 'Gaziantep', 'Gaziantep', NULL ],
			'TR28' => [ 'Giresun', 'Giresun', NULL ],
			'TR29' => [ 'Gümüşhane', 'Gümüşhane', NULL ],
			'TR30' => [ 'Hakkari', 'Hakkari', NULL ],
			'TR31' => [ 'Hatay', 'Hatay', NULL ],
			'TR32' => [ 'Isparta', 'Isparta', NULL ],
			'TR33' => [ 'Mersin', 'Mersin', NULL ],
			'TR34' => [ 'İstanbul', 'İstanbul', NULL ],
			'TR35' => [ 'İzmir', 'İzmir', NULL ],
			'TR36' => [ 'Kars', 'Kars', NULL ],
			'TR37' => [ 'Kastamonu', 'Kastamonu', NULL ],
			'TR38' => [ 'Kayseri', 'Kayseri', NULL ],
			'TR39' => [ 'Kırklareli', 'Kırklareli', NULL ],
			'TR40' => [ 'Kırşehir', 'Kırşehir', NULL ],
			'TR41' => [ 'Kocaeli', 'Kocaeli', NULL ],
			'TR42' => [ 'Konya', 'Konya', NULL ],
			'TR43' => [ 'Kütahya', 'Kütahya', NULL ],
			'TR44' => [ 'Malatya', 'Malatya', NULL ],
			'TR45' => [ 'Manisa', 'Manisa', NULL ],
			'TR46' => [ 'Kahramanmaraş', 'Kahramanmaraş', NULL ],
			'TR47' => [ 'Mardin', 'Mardin', NULL ],
			'TR48' => [ 'Muğla', 'Muğla', NULL ],
			'TR49' => [ 'Muş', 'Muş', NULL ],
			'TR50' => [ 'Nevşehir', 'Nevşehir', NULL ],
			'TR51' => [ 'Niğde', 'Niğde', NULL ],
			'TR52' => [ 'Ordu', 'Ordu', NULL ],
			'TR53' => [ 'Rize', 'Rize', NULL ],
			'TR54' => [ 'Sakarya', 'Sakarya', NULL ],
			'TR55' => [ 'Samsun', 'Samsun', NULL ],
			'TR56' => [ 'Siirt', 'Siirt', NULL ],
			'TR57' => [ 'Sinop', 'Sinop', NULL ],
			'TR58' => [ 'Sivas', 'Sivas', NULL ],
			'TR59' => [ 'Tekirdağ', 'Tekirdağ', NULL ],
			'TR60' => [ 'Tokat', 'Tokat', NULL ],
			'TR61' => [ 'Trabzon', 'Trabzon', NULL ],
			'TR62' => [ 'Tunceli', 'Tunceli', NULL ],
			'TR63' => [ 'Şanlıurfa', 'Şanlıurfa', NULL ],
			'TR64' => [ 'Uşak', 'Uşak', NULL ],
			'TR65' => [ 'Van', 'Van', NULL ],
			'TR66' => [ 'Yozgat', 'Yozgat', NULL ],
			'TR67' => [ 'Zonguldak', 'Zonguldak', NULL ],
			'TR68' => [ 'Aksaray', 'Aksaray', NULL ],
			'TR69' => [ 'Bayburt', 'Bayburt', NULL ],
			'TR70' => [ 'Karaman', 'Karaman', NULL ],
			'TR71' => [ 'Kırıkkale', 'Kırıkkale', NULL ],
			'TR72' => [ 'Batman', 'Batman', NULL ],
			'TR73' => [ 'Şırnak', 'Şırnak', NULL ],
			'TR74' => [ 'Bartın', 'Bartın', NULL ],
			'TR75' => [ 'Ardahan', 'Ardahan', NULL ],
			'TR76' => [ 'Iğdır', 'Iğdır', NULL ],
			'TR77' => [ 'Yalova', 'Yalova', NULL ],
			'TR78' => [ 'Karabük', 'Karabük', NULL ],
			'TR79' => [ 'Kilis', 'Kilis', NULL ],
			'TR80' => [ 'Osmaniye', 'Osmaniye', NULL ],
			'TR81' => [ 'Düzce', 'Düzce', NULL ],
		],
		// Tanzania.
		Country_Code::TANZANIA => [],
		// Uganda.
		Country_Code::UGANDA => [],
		// United States Minor Outlying Islands.
		'UM' => [],
		// United States.
		Country_Code::UNITED_STATES => [
			'AL' => [ 'AL', 'Alabama', NULL ],
			'AK' => [ 'AK', 'Alaska', NULL ],
			'AZ' => [ 'AZ', 'Arizona', NULL ],
			'AR' => [ 'AR', 'Arkansas', NULL ],
			'CA' => [ 'CA', 'California', NULL ],
			'CO' => [ 'CO', 'Colorado', NULL ],
			'CT' => [ 'CT', 'Connecticut', NULL ],
			'DE' => [ 'DE', 'Delaware', NULL ],
			'DC' => [ 'DC', 'District of Columbia', NULL ],
			'FL' => [ 'FL', 'Florida', NULL ],
			'GA' => [ 'GA', 'Georgia', NULL ],
			'HI' => [ 'HI', 'Hawaii', NULL ],
			'ID' => [ 'ID', 'Idaho', NULL ],
			'IL' => [ 'IL', 'Illinois', NULL ],
			'IN' => [ 'IN', 'Indiana', NULL ],
			'IA' => [ 'IA', 'Iowa', NULL ],
			'KS' => [ 'KS', 'Kansas', NULL ],
			'KY' => [ 'KY', 'Kentucky', NULL ],
			'LA' => [ 'LA', 'Louisiana', NULL ],
			'ME' => [ 'ME', 'Maine', NULL ],
			'MD' => [ 'MD', 'Maryland', NULL ],
			'MA' => [ 'MA', 'Massachusetts', NULL ],
			'MI' => [ 'MI', 'Michigan', NULL ],
			'MN' => [ 'MN', 'Minnesota', NULL ],
			'MS' => [ 'MS', 'Mississippi', NULL ],
			'MO' => [ 'MO', 'Missouri', NULL ],
			'MT' => [ 'MT', 'Montana', NULL ],
			'NE' => [ 'NE', 'Nebraska', NULL ],
			'NV' => [ 'NV', 'Nevada', NULL ],
			'NH' => [ 'NH', 'New Hampshire', NULL ],
			'NJ' => [ 'NJ', 'New Jersey', NULL ],
			'NM' => [ 'NM', 'New Mexico', NULL ],
			'NY' => [ 'NY', 'New York', NULL ],
			'NC' => [ 'NC', 'North Carolina', NULL ],
			'ND' => [ 'ND', 'North Dakota', NULL ],
			'OH' => [ 'OH', 'Ohio', NULL ],
			'OK' => [ 'OK', 'Oklahoma', NULL ],
			'OR' => [ 'OR', 'Oregon', NULL ],
			'PA' => [ 'PA', 'Pennsylvania', NULL ],
			'RI' => [ 'RI', 'Rhode Island', NULL ],
			'SC' => [ 'SC', 'South Carolina', NULL ],
			'SD' => [ 'SD', 'South Dakota', NULL ],
			'TN' => [ 'TN', 'Tennessee', NULL ],
			'TX' => [ 'TX', 'Texas', NULL ],
			'UT' => [ 'UT', 'Utah', NULL ],
			'VT' => [ 'VT', 'Vermont', NULL ],
			'VA' => [ 'VA', 'Virginia', NULL ],
			'WA' => [ 'WA', 'Washington', NULL ],
			'WV' => [ 'WV', 'West Virginia', NULL ],
			'WI' => [ 'WI', 'Wisconsin', NULL ],
			'WY' => [ 'WY', 'Wyoming', NULL ],
			'AA' => [ 'AA', 'Armed Forces (AA)', NULL ],
			'AE' => [ 'AE', 'Armed Forces (AE)', NULL ],
			'AP' => [ 'AP', 'Armed Forces (AP)', NULL ],
			//[ 'AS', 'American Samoa', NULL ],
			//[ 'GU', 'Guam', NULL ],
			//[ 'MH', 'Marshall Islands', NULL ],
			//[ 'FM', 'Micronesia', NULL ],
			//[ 'MP', 'Northern Mariana Islands', NULL ],
			//[ 'PW', 'Palau', NULL ],
			//[ 'PR', 'Puerto Rico', NULL ],
			//[ 'VI', 'Virgin Islands', NULL ],
		],
		// Vietnam.
		Country_Code::VIETNAM => [],
		// Mayotte.
		'YT' => [],
		// South Africa.
		Country_Code::SOUTH_AFRICA => [],
		// Zambia.
		Country_Code::ZAMBIA => [],
	];
	// phpcs:enable
}
